import gi
from gi.repository import Gtk, Gio, Gdk, GLib

from hiprompt_gtk.proto import PromptMode


class Window:
    def __init__(self, application, session):
        self.application = application
        self.session = session

        css = Gio.resources_lookup_data("/himitsu/assets/style.css", 0)
        self.provider = Gtk.CssProvider()
        self.provider.load_from_data(css.get_data())
        Gtk.StyleContext.add_provider_for_screen(Gdk.Screen.get_default(),
                                                 self.provider,
                                                 Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)

        self.window = Gtk.Window()
        self.header = Gtk.HeaderBar()
        self.window.set_titlebar(self.header)
        self.window.connect('destroy', self.on_close)
        self.box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        self.window.add(self.box)

        self.deny = None
        self.grant = None
        self.entry = None
        self.buttons = []

        self.accel = Gtk.AccelGroup()
        self.accel.connect(Gdk.keyval_from_name('Escape'), 0, 0, self.on_deny)
        self.window.add_accel_group(self.accel)

        self.make_ui()

    def make_ui(self):
        if self.session.mode == PromptMode.disclose:
            self.make_disclose_ui()
        elif self.session.mode == PromptMode.delete:
            self.make_delete_ui()
        elif self.session.mode is None:
            assert self.session.unlock
        else:
            raise ValueError("Unexpected mode")

        if self.session.unlock:
            self.make_unlock_ui()

        self.header.show_all()
        self.box.show_all()
        GLib.timeout_add(700, self.enable_actions)

    def enable_actions(self, *args):
        for button in self.buttons:
            button.set_sensitive(True)
            button.grab_focus()

    def on_close(self, *args):
        Gtk.main_quit()

    def on_deny(self, *args):
        self.session.status = 1
        Gtk.main_quit()

    def on_unlock_grant(self, *args):
        password = self.entry.get_text()
        self.deny.set_sensitive(False)
        self.grant.set_sensitive(False)
        self.entry.set_sensitive(False)
        self.entry.set_text("Verifying...")
        self.entry.set_visibility(True)
        self.window.remove_accel_group(self.accel)
        confirmed = len(self.session.keys) > 0

        def _callback(status):
            if status:
                if not self.session.mode or confirmed:
                    self.session.status = 0
                    Gtk.main_quit()
                else:
                    self.make_ui()
            else:
                self.entry.set_text("")
                self.entry.set_visibility(False)
                self.entry.set_sensitive(True)
                self.grant.set_sensitive(True)
                self.window.add_accel_group(self.accel)
                self.entry.grab_focus()

        self.session.password(password, _callback)

    def on_disclose_grant(self, *args):
        self.session.status = 0
        Gtk.main_quit()

    def make_unlock_ui(self):
        for child in self.box:
            self.box.remove(child)
        for child in self.header:
            self.header.remove(child)

        self.header.set_title("Unlock keyring")

        if len(self.session.keys) == 0:
            self.deny = Gtk.Button("Cancel")
        else:
            self.deny = Gtk.Button("Deny access")
        self.deny.connect("clicked", self.on_deny)
        self.header.pack_start(self.deny)

        self.grant = Gtk.Button("Grant access")
        self.grant.get_style_context().add_class("destructive-action")
        self.grant.connect("clicked", self.on_unlock_grant)
        self.header.pack_end(self.grant)

        scroller = Gtk.ScrolledWindow()
        scroller.set_propagate_natural_height(True)
        scroller.set_propagate_natural_width(True)
        scroller.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.NEVER)
        self.box.add(scroller)

        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.get_style_context().add_class("mainbox")
        scroller.add(box)

        if len(self.session.keys) == 0:
            label = Gtk.Label("An application has requested to unlock "
                              "your keyring.")
            box.add(label)
        else:
            label = Gtk.Label("An application has requested to unlock "
                              "your keyring.\n"
                              "If you consent, the application will have access to "
                              "the following information:")
            box.add(label)
            box.add(self.make_key_box())

        listbox = Gtk.ListBox()
        listbox.get_style_context().add_class("content")
        listbox.set_selection_mode(Gtk.SelectionMode.NONE)
        box.add(listbox)

        row = Gtk.Box(spacing=8)
        listbox.insert(row, -1)

        rowlabel = Gtk.Label("Password")
        row.pack_start(rowlabel, False, False, 0)

        self.entry = Gtk.Entry()
        row.pack_end(self.entry, True, True, 0)
        self.entry.grab_focus()
        self.entry.set_input_purpose(Gtk.InputPurpose.PASSWORD)
        self.entry.set_visibility(False)
        self.entry.connect('activate', self.on_unlock_grant)

    def make_disclose_ui(self):
        for child in self.box:
            self.box.remove(child)
        for child in self.header:
            self.header.remove(child)

        self.header.set_title("Review secret access request")

        deny = Gtk.Button("Deny access")
        deny.connect("clicked", self.on_deny)
        self.header.pack_start(deny)

        grant = Gtk.Button("Grant access")
        grant.get_style_context().add_class("destructive-action")
        grant.set_sensitive(False)
        self.buttons.append(grant)
        grant.connect("clicked", self.on_disclose_grant)
        self.header.pack_end(grant)

        scroller = Gtk.ScrolledWindow()
        scroller.set_propagate_natural_height(True)
        scroller.set_propagate_natural_width(True)
        scroller.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.NEVER)
        self.box.add(scroller)

        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.get_style_context().add_class("mainbox")
        scroller.add(box)

        label = Gtk.Label("An application has requested permission "
                          "to view your private account information.\n"
                          "If you consent, the application will have access to "
                          "the following information:")
        box.add(label)
        box.add(self.make_key_box())

    def make_delete_ui(self):
        for child in self.box:
            self.box.remove(child)
        for child in self.header:
            self.header.remove(child)

        self.header.set_title("Delete key request")

        deny = Gtk.Button("Cancel")
        deny.connect("clicked", self.on_deny)
        self.header.pack_start(deny)

        grant = Gtk.Button("Confirm deletion")
        grant.get_style_context().add_class("destructive-action")
        grant.connect("clicked", self.on_disclose_grant)
        self.buttons.append(grant)
        grant.set_sensitive(False)
        self.header.pack_end(grant)

        scroller = Gtk.ScrolledWindow()
        scroller.set_propagate_natural_height(True)
        scroller.set_propagate_natural_width(True)
        scroller.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.NEVER)
        self.box.add(scroller)

        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.get_style_context().add_class("mainbox")
        scroller.add(box)

        label = Gtk.Label("An application has requested to remove "
                          "the following keys from your store:")
        box.add(label)
        box.add(self.make_key_box())

    def make_key_box(self):
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)

        for key in self.session.keys:
            listbox = Gtk.ListBox()
            listbox.get_style_context().add_class("content")
            listbox.set_selection_mode(Gtk.SelectionMode.NONE)
            box.add(listbox)

            for pair in key:
                row = Gtk.Box(spacing=6)
                listbox.insert(row, -1)

                name = Gtk.Label(pair.key, xalign=0.0)
                name.get_style_context().add_class("heading")
                row.pack_start(name, True, True, 0)

                if pair.value is not None and not pair.private:
                    pair_value = pair.value
                    if len(pair_value) > 50:
                        pair_value = pair_value[:50] + "..."
                    value = Gtk.Label(pair_value, xalign=1.0)
                    value.get_style_context().add_class("value")
                    row.pack_start(value, True, True, 0)
                elif pair.private:
                    value = Gtk.Label("(not shown)", xalign=1.0)
                    context = value.get_style_context()
                    context.add_class("value")
                    context.add_class("secret")
                    context.add_class("dim-label")
                    row.pack_start(value, True, True, 0)
                else:
                    value = Gtk.Label("(empty)", xalign=1.0)
                    context = value.get_style_context()
                    context.add_class("value")
                    context.add_class("empty")
                    context.add_class("dim-label")
                    row.pack_start(value, True, True, 0)

        return box
