import json
import os

import gi

from himitsu_gtk.himitsu import Himitsu
from himitsu_gtk.key import Key

gi.require_version('Gtk', '3.0')
from gi.repository import Gtk, GLib, GObject, Gio, Gdk, GLib, GdkPixbuf

gi.require_version('Handy', '1')
from gi.repository import Handy


class KeyringWindow:
    def __init__(self, application, args, version):
        self.application = application
        self.args = args
        self.version = version

        Handy.init()

        builder = Gtk.Builder()
        builder.add_from_resource('/nl/brixit/Keyring/ui/main.glade')
        builder.connect_signals(self)
        self.builder = builder
        css = Gio.resources_lookup_data("/nl/brixit/Keyring/ui/style.css", 0)
        self.provider = Gtk.CssProvider()
        self.provider.load_from_data(css.get_data())

        self.provider = Gtk.CssProvider()
        self.provider.load_from_data(css.get_data())

        self.window = builder.get_object("window")
        self.window.set_application(self.application)
        self.mainstack = builder.get_object("mainstack")
        self.statusstack = builder.get_object("statusstack")
        self.headerstack = builder.get_object("headerstack")
        self.leaflet = builder.get_object("leaflet")
        self.keylist = builder.get_object("keylist")
        self.keyinfo = builder.get_object("keyinfo")
        self.addkey = builder.get_object("addkey")
        self.toastrevealer = builder.get_object("toastrevealer")
        self.copyhelp = builder.get_object("copyhelp")

        searchbox = builder.get_object('search')
        searchbox.grab_focus()

        self.apply_css(self.window, self.provider)

        self.window.show()
        self.key = None
        self.touchscreen = False

        try:
            self.himitsu = Himitsu()
            self.load_keys()
        except ConnectionError:
            self.statusstack.set_visible_child_name('error')
        self.add_entry_key = []
        self.add_entry_val = []
        self.search = []

        self.keylist.set_filter_func(self.listbox_filter, None)

        Gtk.main()

    def apply_css(self, widget, provider):
        Gtk.StyleContext.add_provider(widget.get_style_context(),
                                      provider,
                                      Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)

        if isinstance(widget, Gtk.Container):
            widget.forall(self.apply_css, provider)

    def empty(self, widget):
        for child in widget:
            widget.remove(child)

    def load_keys(self):

        titles = {
            'ssh': 'SSH',
            'imap': 'IMAP',
            'smtp': 'SMTP',
            'xmpp': 'XMPP',
            'irc': 'IRC',
        }
        try:
            keys = self.himitsu.query()
        except ConnectionError:
            self.statusstack.set_visible_child_name('error')
            return

        self.empty(self.keylist)
        grouped = {}
        for key in keys:
            proto = key['proto']

            if proto not in grouped:
                grouped[proto] = []
            grouped[proto].append(key)

        for group in grouped:
            if group is None:
                continue
            if group in titles:
                label = titles[group]
            else:
                label = group.title()
            self._make_group(label, grouped[group])

        if None in grouped:
            self._make_group("Unsorted", grouped[None])

        self.statusstack.set_visible_child_name('connected')
        self.keylist.show_all()

    def _make_group(self, name, keys):
        row = Gtk.ListBoxRow()
        row.set_selectable(False)
        row.set_activatable(False)
        label = Gtk.Label(label=name)
        label.get_style_context().add_class('heading')
        label.set_margin_top(8)
        row.add(label)
        self.keylist.add(row)
        row = Gtk.ListBoxRow()
        row.set_selectable(False)
        row.set_activatable(False)
        row.add(Gtk.Separator())
        self.keylist.add(row)

        for key in keys:
            name, detail = self._describe_key(key)
            box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
            label = Gtk.Label(label=name, xalign=0.0)
            label.set_line_wrap(True)
            box.add(label)
            if detail is not None:
                detaillabel = Gtk.Label(label=detail, xalign=0.0)
                detaillabel.set_line_wrap(True)
                detaillabel.get_style_context().add_class('dim-label')
                box.add(detaillabel)
            box.set_margin_top(8)
            box.set_margin_bottom(8)
            box.set_margin_start(10)
            box.set_margin_end(10)
            row = Gtk.ListBoxRow()
            row.add(box)
            row.key = key
            self.keylist.add(row)

    def _describe_key(self, key):
        detail = []
        for pair in key:
            if pair.value is None:
                continue
            if pair.private:
                continue
            if pair.optional:
                continue
            detail.append(f'{pair.key}={pair.value}')
        detail = ' '.join(detail)

        if key['host'] is not None:
            label = key['host']
            if key['username']:
                label = f'{key["username"]}@{label}'
            return label, detail
        else:
            return detail, None

    def show_key(self):
        self.empty(self.keyinfo)
        for pair in self.key:
            if pair.key == 'key':
                continue

            row = Gtk.Box(spacing=6)
            row.set_margin_top(8)
            row.set_margin_bottom(8)
            row.set_margin_start(8)
            row.set_margin_end(8)
            lbr = Gtk.ListBoxRow()
            eb = Gtk.EventBox()
            eb.add(row)
            lbr.add(eb)
            lbr.pair = pair

            lbr.longpress = Gtk.GestureLongPress.new(eb)
            lbr.longpress.connect("pressed", self.on_key_row_longpress)
            lbr.longpress.row = lbr

            self.keyinfo.insert(lbr, -1)

            name = Gtk.Label(pair.key, xalign=0.0)
            name.get_style_context().add_class("heading")
            row.pack_start(name, True, True, 0)

            if pair.value is not None:
                value = Gtk.Label(pair.value, xalign=1.0)
                value.get_style_context().add_class("value")
                row.pack_start(value, True, True, 0)
            elif pair.private:
                value = Gtk.Label("(not shown)", xalign=1.0)
                context = value.get_style_context()
                context.add_class("value")
                context.add_class("secret")
                context.add_class("dim-label")
                row.pack_start(value, True, True, 0)
            else:
                value = Gtk.Label("(empty)", xalign=1.0)
                context = value.get_style_context()
                context.add_class("value")
                context.add_class("empty")
                context.add_class("dim-label")
                row.pack_start(value, True, True, 0)
        self.keyinfo.show_all()
        self.leaflet.set_visible_child_name("main")

    def on_keyrow_activated(self, widget, row):
        self.mainstack.set_visible_child_name('keyinfo')
        self.key = row.key
        self.show_key()

    def on_decrypt_clicked(self, widget, *args):
        result = self.himitsu.query(self.key, decrypt=True)
        self.key = result[0]
        self.show_key()

    def on_delete_clicked(self, widget, *args):
        dialog = Gtk.MessageDialog(
            transient_for=self.window,
            flags=0,
            message_type=Gtk.MessageType.QUESTION,
            buttons=Gtk.ButtonsType.YES_NO,
            text="Are you sure you want to remove this key?",
        )
        dialog.format_secondary_text(str(self.key))
        response = dialog.run()
        if response == Gtk.ResponseType.YES:
            self.himitsu.delete(self.key)
            self.load_keys()
            self.mainstack.set_visible_child_name('nokey')
        dialog.destroy()

    def on_key_row_longpress(self, widget, x, y):
        self.on_key_row_activate(widget, widget.row)

    def on_key_row_activate(self, widget, row):
        pair = row.pair
        if pair.value is not None:
            value = pair.value
        elif pair.private:
            result = self.himitsu.query(self.key, decrypt=True)
            value = result[0][pair.key]
        else:
            return

        clipboard = Gtk.Clipboard.get(Gdk.SELECTION_CLIPBOARD)
        clipboard.set_text(value, -1)

        self.toastrevealer.set_reveal_child(True)
        GLib.timeout_add_seconds(1.4, self.hide_toast)

    def hide_toast(self, *args):
        self.toastrevealer.set_reveal_child(False)
        return False

    def addkey_add_placeholder(self, name=None, value=None):
        row = Gtk.ListBoxRow()
        box = Gtk.Box(spacing=6)
        row.add(box)

        keyentry = Gtk.Entry()
        keyentry.set_width_chars(10)
        if name:
            keyentry.set_text(name)
        keyentry.connect('changed', self.on_add_keypress)
        self.add_entry_key.append(keyentry)

        valentry = Gtk.Entry()
        if value:
            valentry.set_text(value)
        valentry.connect('changed', self.on_add_keypress)
        self.add_entry_val.append(valentry)

        box.pack_start(keyentry, False, False, 0)
        box.pack_end(valentry, True, True, 0)
        self.addkey.insert(row, -1)

    def on_add_clicked(self, widget, *args):
        self.mainstack.set_visible_child_name('add')
        self.empty(self.addkey)
        self.add_entry_key = []
        self.add_entry_val = []
        self.addkey_add_placeholder('proto')
        self.addkey_add_placeholder()
        self.addkey.show_all()

        self.leaflet.set_visible_child_name("main")
        self.add_entry_val[0].grab_focus()

    def on_add_keypress(self, *args):
        has_empty = False
        for entry in self.add_entry_key:
            if entry.get_text() == "":
                has_empty = True
        if not has_empty:
            self.addkey_add_placeholder()
            self.addkey.show_all()

    def on_save_clicked(self, widget, *args):
        newkey = Key()
        for i, key in enumerate(self.add_entry_key):
            key = key.get_text()
            val = self.add_entry_val[i].get_text()

            if key.strip() == "":
                continue

            private = False
            if key.endswith('!'):
                private = True

            newkey.add(key, val, private=private)
        self.himitsu.add(newkey)
        self.leaflet.set_visible_child_name("sidebar")
        self.load_keys()

    def on_search_changed(self, widget):
        query = widget.get_text()
        component = query.split()
        self.search = []
        for c in component:
            if '=' in c:
                key, val = c.split('=', maxsplit=1)
                self.search.append((key, val))
            else:
                self.search.append(c)
        self.keylist.invalidate_filter()

    def listbox_filter(self, row, user_data):
        if len(self.search) == 0:
            return True

        if not hasattr(row, 'key'):
            return True
        key = row.key
        for component in self.search:
            if isinstance(component, tuple):
                val = key[component[0]]
                if val != component[1]:
                    return False
            else:
                for pair in key:
                    if pair.value is not None and component in pair.value:
                        break
                else:
                    return False
        return True

    def on_back_clicked(self, widget, *args):
        self.headerstack.set_visible_child_name('sidebar')
        self.leaflet.set_visible_child_name('sidebar')

    def on_leaflet_change(self, *args):
        folded = self.leaflet.get_folded()
        main = self.leaflet.get_visible_child_name() == "main"
        self.headerstack.set_visible_child_name("main" if main and folded else "sidebar")

    def on_main_window_destroy(self, widget):
        Gtk.main_quit()

    def on_touch_event(self, *args):
        if not self.touchscreen:
            self.touchscreen = True
            self.copyhelp.set_text("Hold a row to copy the value to the clipboard")
