import os
import socket

from himitsu_gtk.key import Key


class Himitsu:
    def __init__(self, path=None):
        self.path = path
        if self.path is None:
            self.path = os.path.join(os.getenv('XDG_RUNTIME_DIR'), 'himitsu')

        if not os.path.exists(self.path):
            raise ConnectionError(f"Himitsu socket not found: {self.path}")

    def _readline(self, s):
        buffer = b''
        while True:
            char = s.recv(1)
            if char == b'\n':
                return buffer
            buffer += char

    def _cmd(self, cmd):
        s = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        s.connect(self.path)
        s.sendall(f"{cmd}\n".encode())

        result = []
        while True:
            line = self._readline(s).decode('utf-8')
            if line == 'end':
                break
            result.append(Key(line))
        return result

    def query(self, key=None, decrypt=False):
        cmd = ["query"]
        if decrypt:
            cmd += ["-d"]
        if isinstance(key, Key):
            for pair in key:
                if pair.key == "key":
                    continue
                if pair.private:
                    continue
                if pair.value is None:
                    continue
                cmd += [f"{pair.key}={pair.value}"]
        elif isinstance(key, dict):
            for k in key:
                cmd += [f"{k}={key[k]}"]
        return self._cmd(" ".join(cmd))

    def delete(self, key):
        cmd = ["del"]
        if isinstance(key, Key):
            for pair in key:
                if pair.key == "key":
                    continue
                if pair.private:
                    continue
                if pair.value is None:
                    continue
                cmd += [f"{pair.key}={pair.value}"]
        elif isinstance(key, dict):
            for k in key:
                cmd += [f"{k}={key[k]}"]
        return self._cmd(" ".join(cmd))

    def add(self, key):
        cmd = ["add"]
        if isinstance(key, Key):
            for pair in key:
                if pair.key == "key":
                    continue
                cmd += [f"{pair.key}={pair.value}"]
        elif isinstance(key, dict):
            for k in key:
                cmd += [f"{k}={key[k]}"]
        return self._cmd(" ".join(cmd))


if __name__ == '__main__':
    test = Himitsu()
    for k in test.query():
        print(k)
